/*
 * This applet provides a graphical user interface for the SNOW
 * whitespace steganography algorithm. It enables a secret message
 * to be compressed and encrypted, then embedded in a cover message
 * in the form of trailing whitespace (spaces and tabs). It also
 * allows embedded messages to be extracted.
 *
 * Written by Matthew Kwan - September 1997
 */

import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringReader;
import java.io.BufferedReader;
import java.io.FileNotFoundException;

import java.awt.Font;
import java.awt.Frame;
import java.awt.Label;
import java.awt.Panel;
import java.awt.Button;
import java.awt.TextArea;
import java.awt.Dimension;
import java.awt.TextField;
import java.awt.FlowLayout;
import java.awt.FileDialog;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.applet.Applet;


/*
 * Top level class.
 * Builds the user interface and handles all events.
 */

public class SnowFront
	extends java.applet.Applet
	implements ActionListener, Runnable
{
	private TextField	password_field;
	private TextArea	hidden_text;
	private TextArea	cover_text;
	private Label		status_label;

		// This class implements a panel with vertical layout.
	class VPanel extends Panel {
		private Dimension	prefSize;

		VPanel (int width, int height) {
		    setLayout (new FlowLayout (FlowLayout.LEFT));
		    prefSize = new Dimension (width, height);
		}

		public Dimension	getPreferredSize () {
		    return (prefSize);
		}
	}

		// Starting point for the applet.
		// Builds the user interface.
	public void	init () {
	    Panel	hpanel;
	    VPanel	vpanel;
	    Button	b;
	    Font	text_font = new Font ("TimesRoman", Font.PLAIN, 12);

	    vpanel = new VPanel (600, 600);
	    vpanel.setFont (new Font ("TimesRoman", Font.BOLD, 12));

			// First row. Hidden message controls.
	    hpanel = new Panel ();
	    hpanel.add (new Label ("Hidden message"));

	    b = new Button ("Embed");
	    b.addActionListener (this);
	    hpanel.add (b);

	    b = new Button ("Extract");
	    b.addActionListener (this);
	    hpanel.add (b);

	    vpanel.add (hpanel);

			// Second row. Hidden message text.
	    hidden_text = new TextArea ("", 3, 80,
					TextArea.SCROLLBARS_VERTICAL_ONLY);
	    hidden_text.setFont (text_font);
	    vpanel.add (hidden_text);

			// Third row. Password.
	    hpanel = new Panel ();
	    hpanel.add (new Label ("Password:"));
	    password_field = new TextField (60);
	    password_field.setFont (text_font);
	    password_field.setEchoChar ('*');
	    hpanel.add (password_field);

	    vpanel.add (hpanel);

			// Fourth row. Cover text controls.
	    hpanel = new Panel ();
	    hpanel.add (new Label ("Cover text"));

	    b = new Button ("Load");
	    b.addActionListener (this);
	    hpanel.add (b);

	    b = new Button ("Save");
	    b.addActionListener (this);
	    hpanel.add (b);

	    b = new Button ("Clear");
	    b.addActionListener (this);
	    hpanel.add (b);

	    vpanel.add (hpanel);

			// Fifth row. Cover text.
	    cover_text = new TextArea (20, 80);
	    cover_text.setFont (text_font);
	    vpanel.add (cover_text);

			// Sixth row. Status label.
	    hpanel = new Panel ();
	    hpanel.add (new Label ("Status:"));

	    String	label_name = "None";

	    for (int i=0; i<10; i++)	// Padding to allow for longer messages
		label_name += "          ";

	    status_label = new Label (label_name);
	    status_label.setFont (text_font);
	    hpanel.add (status_label);

	    vpanel.add (hpanel);

	    add (vpanel);
	}

		// This is a button-event driven application, so it
		// doesn't need to be explicitly run.
	public void	run () {}

		// Set the status label.
	public void	set_status_label (String s) {
	    status_label.setText (s);
	}

		// Button event handling.
	public void	actionPerformed (ActionEvent e) {
	    String	command = e.getActionCommand ();

	    if (command == "Embed")
		embed ();
	    else if (command == "Extract")
		extract ();
	    else if (command == "Load")
		load_cover_text ();
	    else if (command == "Save")
		save_cover_text ();
	    else if (command == "Clear")
		cover_text.setText ("");
	}

		// Embed the hidden message in the cover text.
	private void	embed () {
	    String		password = password_field.getText ();
	    BitFilter		bf;
	    BufferedReader	br;
	    SnowEncode		se;

	    br = new BufferedReader (new StringReader (cover_text.getText ()));
	    cover_text.setText ("");
	    bf = se = new SnowEncode (true, null, br, cover_text, 80);
	    if (password != null && password.length() > 0)
		bf = new SnowEncrypt (true, bf, password);
	    bf = new SnowCompress (true, bf);

	    byte	message[] = hidden_text.getText().getBytes();

	    for (int i=0; i<message.length; i++) {
		int	v = message[i];

		for (int j=0; j<8; j++)
		    bf.receive_bit ((v & (128 >> j)) != 0);
	    }

	    bf.flush ();
	    set_status_label (se.statusMessage ());
	}

		// Extract a hidden message from the cover text.
	private void	extract () {
	    BitFilter	bf;
	    String	password = password_field.getText ();

	    hidden_text.setText ("");

	    bf = new SnowOutput (hidden_text);
	    bf = new SnowCompress (false, bf);
	    if (password != null && password.length() > 0)
		bf = new SnowEncrypt (false, bf, password);

	    BufferedReader	br;
	    SnowEncode		se;

	    br = new BufferedReader (new StringReader (cover_text.getText ()));
	    se = new SnowEncode (false, bf, br, null, 80);
	    se.decode ();

	    set_status_label ("Message extracted");
	}

		// Load the cover text from a file.
	private void	load_cover_text () {
	    FileDialog	fd = new FileDialog (new Frame ("Load"),
					"Cover text file", FileDialog.LOAD);

	    fd.show ();

	    String	fname = fd.getFile ();

	    if (fname == null || fname.length() == 0)
		return;

	    String	path = fd.getDirectory() + fname;
	    FileReader	fp;

	    try {
		fp = new FileReader (path);
	    } catch (FileNotFoundException e) {
		set_status_label ("No such file - " + fname);
		return;
	    }

	    char	cbuf[] = new char[4096];
	    int		n;

	    cover_text.setText ("");
	    try {
		while ((n = fp.read (cbuf)) >= 0)
		    cover_text.append (new String (cbuf, 0, n));
		fp.close ();
	    } catch (IOException e) {
		set_status_label ("Could not read file - " + fname);
		return;
	    }

	    set_status_label ("Cover text loaded");
	}

		// Save the cover text to a file.
	private void	save_cover_text () {
	    FileDialog	fd = new FileDialog (new Frame ("Save"),
					"Cover text file", FileDialog.SAVE);

	    fd.show ();

	    String	fname = fd.getFile ();

	    if (fname == null || fname.length () == 0)
		return;

	    String	path = fd.getDirectory () + fname;
	    FileWriter	fp;

	    try {
		fp = new FileWriter (path);
		fp.write (cover_text.getText ());
		fp.close ();
	    } catch (IOException e) {
		set_status_label ("Could not write to file - " + fname);
		return;
	    }

	    set_status_label ("Cover text saved");
	}
}
