/*
 * Command-line program for encrypting/decrypting data files
 * with the ICE encryption algorithm.
 *
 * It uses Cipher Block Chaining (CBC) with an initialization
 * vector obtained by default from the gettimeofday call.
 *
 * Usage: ice [-C][-D][-N][-Z][-p passwd][-l level] [filename ...]
 *
 *	-C : Send encrypted/decrypted output to stdout.
 *	-D : Decrypt the input data.
 *	-N : Do not ask for confirmation for interactive passwords.
 *	-Z : Use a zero initializing vector. Otherwise use time of day.
 *	-p : Specify the password on the command-line.
 *	-l : Specify the ICE level of encryption.
 *
 * For encryption, an encrypted file "infile.ice" will be created for
 * each specified file. It should have the same permission bits as the
 * original files.
 *
 * For decryption, the files must have a ".ice" suffix, or there must
 * be a file "infile.ice" for each file.
 *
 * Written by Matthew Kwan - April 1997
 */

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Date;


class IceCrypt {
	private static final int	MAX_PASSLEN = 1171;

		// Build the key that will encrypt/decrypt the data.
	private static IceKey	key_build (int level, String passwd) {
	    IceKey	ik = new IceKey (level);
	    byte	buf[] = new byte[ik.keySize ()];
	    int		pass_idx, i;
	    byte	passbytes[] = new byte[passwd.length()];

	    passwd.getBytes (0, passwd.length(), passbytes, 0);

	    pass_idx = i = 0;
	    while (pass_idx < passbytes.length) {
		int	c = passbytes[pass_idx] & 0x7f;
		int	idx = i / 8;
		int	offset = i & 7;

		if (offset == 0) {
		    buf[idx] = (byte) (c << 1);
		} else if (offset == 1) {
		    buf[idx] |= c;
		} else {
		    buf[idx] |= (c >>> (offset - 1));

		    if (idx + 1 < ik.keySize ())
			buf[idx + 1] = (byte) (c << (9 - offset));
		}

		pass_idx++;
		i += 7;

		if (i >= 8 * ik.keySize ())
		    break;
	    }

	    ik.set (buf);

	    return (ik);
	}


		// If possible, zero out the contents of a file.
	private static void	file_zero (File f) {
	    FileOutputStream	fs;
	    long		n = f.length ();

	    try {
		fs = new FileOutputStream (f);

		for (long i=0; i<n; i++)
		    fs.write (0);

		fs.close ();
	    } catch (IOException e) {
	    } catch (SecurityException e) {
	    }
	}


		// Erase a file. Fill it with zeroes, then unlink it.
	private static boolean	erase_file (String fname) {
	    File	f = new File (fname);

	    file_zero (f);

	    try {
		f.delete ();
	    } catch (SecurityException e) {
		System.err.println (fname + " - could not remove file");
		return (false);
	    }

	    return (true);
	}


		// Encrypt a data stream with the given key.
		// Uses cipher-block-chaining mode.
		// The first 3 bytes are the characters "ice". The next
		// characte is ASCII '0' plus the ICE level being used.
		// The next 8 bytes are the initializing vector.
	private static boolean	encrypt_stream (
	    IceKey		ik,
	    int			level,
	    Block		iv,
	    InputStream		inf,
	    OutputStream	outf
	) {
	    int		n;
	    Block	buf = new Block ();
	    Block	inbuf = new Block ();
	    Block	outbuf = new Block ();

	    try {
		outf.write ('i');
		outf.write ('c');
		outf.write ('e');
		outf.write ('0' + level);
	    } catch (IOException e) {
		System.err.println ("Failed to write output");
		return (false);
	    }

	    outbuf.copy (iv);
	    if (!outbuf.write (outf))
		return (false);

	    while ((n = inbuf.read (inf)) == 8) {
		inbuf.xor (outbuf);
		ik.encrypt (inbuf.data, outbuf.data);
		if (!outbuf.write (outf))
		    return (false);
	    }

	    ik.encrypt (iv.data, buf.data);	// Generate padding bytes.
	    inbuf.pad (n, buf);
	    inbuf.xor (outbuf);
	    ik.encrypt (inbuf.data, outbuf.data);

	    if (!outbuf.write (outf))
		return (false);

	    try {
		outf.flush ();
	    } catch (IOException e) {
		return (false);
	    }

	    return (true);
	}


		// Encrypt the specified file.
		// If an output stream isn't specified, write the
		// encrypted data to a file "fname.ice", then remove
		// the original file "fname".
	private static boolean	encrypt_file (
	    IceKey		ik,
	    int			level,
	    Block		iv,
	    String		fname,
	    OutputStream	output_file
	) {
	    InputStream		inf;
	    OutputStream	outf;
	    boolean		res;

	    try {
		inf = new FileInputStream (fname);
	    } catch (FileNotFoundException e) {
		System.err.println (fname + " - file not found");
		return (false);
	    } catch (SecurityException e) {
		System.err.println (fname + " - unable to read");
		return (false);
	    }

	    if (output_file == null) {
		try {
		    outf = new FileOutputStream (fname + ".ice");
		} catch (IOException e) {
		    System.err.println (fname + ".ice - unable to write");
		    return (false);
		} catch (SecurityException e) {
		    System.err.println (fname + ".ice - security violation");
		    return (false);
		}
	    } else
		outf = output_file;

	    res = encrypt_stream (ik, level, iv, inf, outf);

	    try {
		if (output_file == null)
		    outf.close ();
		inf.close ();
	    } catch (IOException e) {
	    }

	    if (!res)
		return (false);

			// Delete the source file if a new file is made.
	    if (output_file == null && !erase_file (fname))
		return (false);

	    return (true);
	}


		// Decrypt a data stream with the given key.
		// The first 3 characters must be "ice", followed by the
		// encryption level + '0'. Next is the initializing vector,
		// followed by the actual encrypted data.
		// The final block read will be padded. Its true length is
		// given in the lower 3 bits of the 7th byte of the block.
	private static boolean	decrypt_stream (
	    String		passwd,
	    InputStream		inf,
	    OutputStream	outf
	) {
	    Block		prevbuf = new Block ();
	    Block		inbuf = new Block ();
	    Block		outbuf = new Block ();
	    byte		magic[] = new byte[4];
	    int			n, level;
	    IceKey		ik;

			// Check the magic number.
	    try {
		n = inf.read (magic, 0, 4);
	    } catch (IOException e) {
		n = 0;
	    }

	    if (n != 4) {
		System.err.println ("Could not read magic number");
		return (false);
	    }

	    if (magic[0] != 'i' || magic[1] != 'c' || magic[2] != 'e') {
		System.err.println ("Illegal magic number");
		return (false);
	    }

	    level = magic[3] - '0';
	    if (level < 0)
		level += 256;

	    if (level > 128) {
		System.err.println ("Level " + level + " is too high (> 128)");
		return (false);
	    }

	    ik = key_build (level, passwd);

	    if (prevbuf.read (inf) != 8) {
		System.err.println ("Could not read initializing vector");
		return (false);
	    }


	    boolean	block_loaded = false;

	    while ((n = inbuf.read (inf)) == 8) {
		if (block_loaded && !outbuf.write (outf))
		    return (false);

		ik.decrypt (inbuf.data, outbuf.data);
		outbuf.xor (prevbuf);
		block_loaded = true;

		prevbuf.copy (inbuf);
	    }

	    if (n != 0)
		System.err.println ("Warning: truncated final block = " + n);

		// The buffer should contain padding info in its last byte
	    n = outbuf.data[7] & 7;
	    if (!outbuf.write (outf, n))
		return (false);

	    try {
		outf.flush ();
	    } catch (IOException e) {
		return (false);
	    }

	    return (true);
	}


		// Decrypt the specified file.
		// If the filename is of the form "foo.ice", that will be the
		// source file. Otherwise, look for a file "filename.ice".
		// If the output stream isn't specified, write to output to
		// a file without the ".ice" suffix, then delete the original
		// file.

	private static boolean	decrypt_file (
	    String		passwd,
	    String		fname,
	    OutputStream	output_file
	) {
	    InputStream		inf;
	    OutputStream	outf;
	    String		infname, outfname;
	    boolean		res;

	    if (fname.endsWith (".ice")) {
		infname = fname;
		outfname = fname.substring (0, fname.length () - 4);
	    } else {
		infname = fname + ".ice";
		outfname = fname;
	    }

	    try {
		inf = new FileInputStream (infname);
	    } catch (FileNotFoundException e) {
		System.err.println (infname + " - file not found");
		return (false);
	    } catch (SecurityException e) {
		System.err.println (infname + " - unable to read");
		return (false);
	    }

	    if (output_file == null) {
		try {
		    outf = new FileOutputStream (outfname);
		} catch (IOException e) {
		    System.err.println (outfname + " - unable to write");
		    return (false);
		} catch (SecurityException e) {
		    System.err.println (outfname + " - security violation");
		    return (false);
		}
	    } else
		outf = output_file;

	    res = decrypt_stream (passwd, inf, outf);

	    try {
		if (output_file == null)
		    outf.close ();
		inf.close ();
	    } catch (IOException e) {
	    }

	    if (!res)
		return (false);

	    if (output_file == null && !erase_file (infname))
		return (false);

	    return (true);
	}


		// Read in a string from standard input.
		// Java doesn't seem to support echo suppression.
	private static String	string_read () {
	    byte	passbuf[] = new byte[MAX_PASSLEN];
	    int		i;

	    for (i=0; i<MAX_PASSLEN; i++) {
		int	c;

		try {
		    c = System.in.read ();
		} catch (IOException e) {
		    c = -1;
		}

		if (c == '\n' || c < 0)
		    break;

		passbuf[i] = (byte) c;
	    }

	    if (i == 0)
		return (null);
	    else
		return (new String (passbuf, 0, 0, i));
	}


		// Prompt the user for a password.
		// Ask for confirmation if required.
	private static String	password_read (boolean confirm_flag) {
	    String	passwd;

	    System.out.print ("Password: ");
	    System.out.flush ();
	    if ((passwd = string_read ()) == null) {
		System.err.println ("No password entered");
		return (null);
	    }

	    if (confirm_flag) {
		String		passwd2;

		System.out.print ("Retype password: ");
		System.out.flush ();
		passwd2 = string_read ();

		if (passwd.compareTo (passwd2) != 0) {
		    System.err.println ("Password mismatch");
		    return (null);
		}
	    }

	    return (passwd);
	}


		// Build the initial value block.
		// If it isn't zero, set it from the time of day.
	private static void	build_iv (Block iv, boolean zero_flag) {
	    if (zero_flag) {
		for (int i=0; i<8; i++)
		    iv.data[i] = 0;
	    } else {
		Date	d = new Date ();
		long	v = d.getTime ();

		for (int i=0; i<8; i++) {
		    iv.data[i] = (byte) (v & 0xff);
		    v >>>= 8;
		}
	    }
	}


		// Entry point to the program.
		// Parse the command-line arguments and start things running.
	public static void	main (String argv[]) {
	    int			optind;
	    boolean		errflag = false;
	    boolean		confirm_flag = true;
	    boolean		decrypt_flag = false;
	    boolean		zero_iv_flag = false;
	    String		passwd_string = null;
	    int			level = 1;
	    OutputStream	dest_fp = null;

	    for (optind = 0; optind < argv.length
				&& argv[optind].charAt(0) == '-'; optind++) {
		String	optarg;

		if (argv[optind].length() < 2) {
		    errflag = true;
		    break;
		}

		switch (argv[optind].charAt(1)) {
		    case 'C':
			dest_fp = System.out;
			break;
		    case 'D':
			decrypt_flag = true;
			break;
		    case 'N':
			confirm_flag = false;
			break;
		    case 'Z':
			zero_iv_flag = true;
			break;
		    case 'l':
			if (argv[optind].length() > 2)
			    optarg = argv[optind].substring(2);
			else if (++optind == argv.length) {
			    errflag = true;
			    break;
			} else
			    optarg = argv[optind];

			try {
			    level = Integer.parseInt (optarg);
			} catch (NumberFormatException e) {
			    System.err.println ("Illegal value: " + optarg);
			    errflag = true;
			    break;
			}
			break;
		    case 'p':
			if (argv[optind].length() > 2)
			    optarg = argv[optind].substring(2);
			else if (++optind == argv.length) {
			    errflag = true;
			    break;
			} else
			    optarg = argv[optind];

			passwd_string = optarg;
			break;
		    default:
			System.err.println ("Illegal option: " + argv[optind]);
			errflag = true;
			break;
		}

		if (errflag)
		    break;
	    }

	    if (errflag) {
		System.err.print ("Usage: IceCrypt [-C][-D][-N][-Z]");
		System.err.println ("[-p passwd][-l level][filename ...]");
		System.exit (1);
	    }

	    if (passwd_string == null &&
			(passwd_string = password_read (confirm_flag)) == null)
		System.exit (1);

	    if (decrypt_flag) {
		if (level != 1)
		    System.err.println (
			"Warning: level setting is ignored during decryption");

		if (optind == argv.length) {
		    if (!decrypt_stream (passwd_string, System.in, System.out))
			System.exit (1);
		} else {
		    for (int i=optind; i<argv.length; i++)
			if (!decrypt_file (passwd_string, argv[i], dest_fp))
			    System.exit (1);
		}
	    } else {
		IceKey	ik = key_build (level, passwd_string);
		Block	iv = new Block ();

		build_iv (iv, zero_iv_flag);

		if (optind == argv.length) {
		    if (!encrypt_stream (ik, level, iv, System.in, System.out))
			System.exit (1);
		} else {
		    for (int i=optind; i<argv.length; i++)
			if (!encrypt_file (ik, level, iv, argv[i], dest_fp))
			    System.exit (1);
		}
	    }
	}
}
